using System;
using System.Threading;
using System.Runtime.Remoting.Messaging;
using System.Security;
using System.Security.Permissions;
using System.Collections;
using System.Runtime.Serialization;
using System.Runtime.InteropServices;
using System.Security.Principal;
using System.Security.AccessControl;
using Microsoft.Win32.SafeHandles;
using System.Runtime.CompilerServices;

namespace CSharpRecipes
{
	public class ThreadingSync
    {
        #region 18.1 Tworzenie pl statycznych dla konkretnych wtkw
        public static void PerThreadStatic()
        {
            TestStaticField();
        }

        public class Foo
        {
            [ThreadStaticAttribute()]
            public static string bar = "Zainicjowany acuch znakw";
        }

        private static void TestStaticField()
        {
            ThreadStaticField.DisplayStaticFieldValue();

            Thread newStaticFieldThread = 
                new Thread(ThreadStaticField.DisplayStaticFieldValue);

            newStaticFieldThread.Start();

            ThreadStaticField.DisplayStaticFieldValue();
        }

        private class ThreadStaticField
        {
            [ThreadStaticAttribute()]
            public static string bar = "Zainicjowany acuch znakw";

            public static void DisplayStaticFieldValue()
            {
                string msg = 
                    string.Format("{0} zawiera pole statyczne o wartoci: {1}", 
                        Thread.CurrentThread.GetHashCode(),
                        ThreadStaticField.bar);
                Console.WriteLine(msg);
            }
        }


        #endregion

        #region 18.2 Zapewnianie dostpu o bezpiecznych wtkach do skadowych klasy
        public static void ThreadSafeAccess()
        {
            DeadLock deadLock = new DeadLock();
            lock(deadLock)
            {
                Thread thread = new Thread(deadLock.Method1);
                thread.Start();

                // realizacja czasochonnego zadania
            }

            int num = 0;
            if(Monitor.TryEnter(MonitorMethodAccess.SyncRoot,250))
            {
                MonitorMethodAccess.ModifyNumericField(10);
                num = MonitorMethodAccess.ReadNumericField();
                Monitor.Exit(MonitorMethodAccess.SyncRoot);
            }
            Console.WriteLine(num);

        }

        public static class NoSafeMemberAccess
        {
            private static int numericField = 1;

            public static void IncrementNumericField() 
            {
                ++numericField;
            }

            public static void ModifyNumericField(int newValue) 
            {
                numericField = newValue;
            }

            public static int ReadNumericField()
            {
                return (numericField);
            }
        }

        public static class SaferMemberAccess
        {
            private static int numericField = 1;
            private static object syncObj = new object();

            public static void IncrementNumericField() 
            {
                lock(syncObj)
                {
                    ++numericField;
                }
            }

            public static void ModifyNumericField(int newValue) 
            {
                lock (syncObj)
                {
                    numericField = newValue;
                }
            }

            public static int ReadNumericField()
            {
                lock (syncObj)
                {
                    return (numericField);
                }
            }
        }

        public class DeadLock
        {
            private object syncObj = new object();

            public void Method1()
            {
                lock(syncObj)
                {
                    // dalsze czynnoci
                }
            }
        }

        public static class MonitorMethodAccess
        {
            private static int numericField = 1;
            private static object syncObj = new object();

            public static object SyncRoot
            {
                get { return syncObj; }
            }

            public static void IncrementNumericField() 
            {
                if (Monitor.TryEnter(syncObj, 250))
                {
                    try
                    {
                        ++numericField;
                    }
                    finally
                    {
                        Monitor.Exit(syncObj);
                    }
                }
            }

            public static void ModifyNumericField(int newValue) 
            {
                if (Monitor.TryEnter(syncObj, 250))
                {
                    try
                    {
                        numericField = newValue;
                    }
                    finally
                    {
                        Monitor.Exit(syncObj);
                    }
                }
            }

            public static int ReadNumericField()
            {
                if (Monitor.TryEnter(syncObj, 250))
                {
                    try
                    {
                        return (numericField);
                    }
                    finally
                    {
                        Monitor.Exit(syncObj);
                    }
                }

                return (-1);
            }
            [MethodImpl (MethodImplOptions.Synchronized)]
            public static void MySynchronizedMethod()
            {
            }

        }

        
        #endregion

        #region 18.3 Zapobieganie cichemu zakoczeniu wtkw
        public static void PreventSilentTermination()
        {
            MainThread mt = new MainThread();
            mt.CreateNewThread();
        }

        public class MainThread
        {
            public void CreateNewThread()
            {
                // uruchomienie nowego wtku, by wykona wspbiene zadania
                Thread newWorkerThread = new Thread(Worker.DoWork);
                newWorkerThread.Start();
            }
        }

        public class Worker
        {
            // metoda wywoywana przez metod delegowan ThreadStart, by wykona zadania wspbiene
            public static void DoWork ()
            {
                try
                {
                    // zadania wykonywane przez wtek
                    throw new Exception("Bum!");
                }
                catch(Exception e) 
                {
                    // obsuga wyjtku wtku
                    Console.WriteLine(e.ToString());
                    // nie wyrzucamy wyjtku ponownie
                }
                finally
                {
                    // porzdki
                }
            }
        }
        #endregion

        #region 18.4 Odpytywanie asynchronicznej metody delegowanej
        public static void PollingAsyncDelegate()
        {
            AsyncAction aa = new AsyncAction();
            aa.PollAsyncDelegate();
        }

        public class AsyncAction
        {
            public void PollAsyncDelegate()
            {
                // utworzenie metody delegowanej
                AsyncInvoke method1 = TestAsyncInvoke.Method1;
                // poniewa wywoanie zwrotne nie jest wykonywane,
                // dla wywoania zwrotnego oraz danych obiektu wywoania zwrotnego
                // przekazywane s wartoci null                
                Console.WriteLine("Wywoanie metody BeginInvoke w wtku {0}", 
                    Thread.CurrentThread.ManagedThreadId);
                IAsyncResult asyncResult = method1.BeginInvoke(null, null);

                Console.WriteLine("Rozpoczcie odpytywania w wtku {0}", 
                    Thread.CurrentThread.ManagedThreadId);
                while (!asyncResult.IsCompleted)
                {
                    // przerwanie pracy na 1 sekund
                    Thread.Sleep(1000);
                    Console.Write('.');
                }
                Console.WriteLine("Zakoczono odpytywanie w wtku {0}", 
                    Thread.CurrentThread.ManagedThreadId);

                try
                {
                    int retVal = method1.EndInvoke(asyncResult);
                    Console.WriteLine("Zwrcona warto: " + retVal);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.ToString());
                }
            }
        }

        public delegate int AsyncInvoke();

        public class TestAsyncInvoke
        {
            public static int Method1()
            {
                Console.WriteLine("Wywoano metod Method1 w wtku {0}",
                    Thread.CurrentThread.ManagedThreadId);
                return (1);
            }
        }

        #endregion

        #region 18.5 Definiowanie czasu wygasania asynchronicznej metody delegowanej
        public static void TimeoutAsyncDelegate()
        {
            AsyncAction1 aa1 = new AsyncAction1();
            aa1.TimeOutWakeAsyncDelegate();
        }

        public class AsyncAction1
        {
            public void TimeOutWakeAsyncDelegate()
            {
                // utworzenie metody delegowanej
                AsyncInvoke method1 = TestAsyncInvoke.Method1;
                // poniewa wywoanie zwrotne nie jest wykonywane,
                // dla wywoania zwrotnego oraz danych obiektu wywoania zwrotnego
                // przekazywane s wartoci null
                Console.WriteLine("Wywoanie metody BeginInvoke w wtku {0}",
                    Thread.CurrentThread.ManagedThreadId);
                IAsyncResult asyncResult = method1.BeginInvoke(null, null);

                int counter = 0;
                while (counter <= 25 && 
                    !asyncResult.AsyncWaitHandle.WaitOne(20, true))
                {
                    counter++;
                    Console.WriteLine("Przetwarzanie...");
                }

                if (asyncResult.IsCompleted)
                {
                    int retVal = method1.EndInvoke(asyncResult);
                    Console.WriteLine("Warto wynikowa (przed czasem wyganicia): " + retVal);
                }
                else
                {
                    Console.WriteLine("Czas wygas");
                }	    
            }
        }

        public delegate int AsyncInvoke1();

        public class TestAsyncInvoke1
        {
            public static int Method1()
            {
                Console.WriteLine("Wywoano metod Method1 w wtku {0}",
                    Thread.CurrentThread.ManagedThreadId);
                return (1);
            }
        }

        #endregion

        #region 18.6 Uzyskiwanie powiadomienia o zakoczeniu dziaania asynchronicznej metody delegowanej
        public static void CompletionAsyncDelegate()
        {
            AsyncAction2 aa2 = new AsyncAction2();
            aa2.CallbackAsyncDelegate();
        }

        public class AsyncAction2
        {
            public void CallbackAsyncDelegate()
            {
                AsyncCallback callBack = DelegateCallback;

                AsyncInvoke method1 = TestAsyncInvoke.Method1;
                Console.WriteLine("Wywoanie metody BeginInvoke w wtku {0}",
                    Thread.CurrentThread.ManagedThreadId);
                IAsyncResult asyncResult = method1.BeginInvoke(callBack, method1);

                // nie trzeba odpytywa ani wywoywa metody WaitOne, wic mona wrci do metody wywoujcej
                return;
            }

            private static void DelegateCallback(IAsyncResult iresult)
            {
                Console.WriteLine("Otrzymanie wywoania zwrotnego w wtku {0}",
                    Thread.CurrentThread.ManagedThreadId);
                AsyncResult asyncResult = (AsyncResult)iresult;
                AsyncInvoke method1 = (AsyncInvoke)asyncResult.AsyncDelegate;

                int retVal = method1.EndInvoke(asyncResult);
                Console.WriteLine("Warto wynikowa (wywoanie zwrotne): " + retVal);
            }
        }

        public delegate int AsyncInvoke2();

        public class TestAsyncInvoke2
        {
            public static int Method1()
            {
                Console.WriteLine("Wywoano metod Method1 w wtku {0}",
                    Thread.CurrentThread.ManagedThreadId);
                return (1);
            }
        }

        #endregion

        #region 18.7 Ustalanie, czy danie skierowane do puli wtkw zostanie zakolejkowane
        public static void DeterminePoolQueue()
        {
            TestThreads.Run();
        }

        public class TestThreads
        {
            public static void Run()
            {
                SpawnManyThreads();
                // trzeba chwil poczeka, bo w przeciwnym razie
                // dziaajce w tle wtki z puli nie zostan uruchomione
                // przed zakoczeniem wtku gwnego                
                Console.WriteLine("Oczekiwanie na zakoczenie gwnego wtku...");
                Thread.Sleep(2000);
                Console.WriteLine("Zakoczenie gwnego wtku...");
            }

            public static bool SpawnManyThreads()
            {
                try
                {
                    for(int i=0;i<500;i++)
                    {
                        // naley chwil poczeka albo pula wtkw
                        // nie udostpni wtkw na danie
                        Thread.Sleep(100);
                        // sprawdzenie, czy w puli dostpne s wtki robocze
                        if(true == IsThreadAvailable(true))
                        {
                            // uruchomienie wtku, jeli pula nie jest pusta
                            Console.WriteLine("Wtek roboczy jest dostpny...");
                            ThreadPool.QueueUserWorkItem(ThreadProc,i);
                        }
                        else
                            Console.WriteLine("Wtek roboczy NIE jest dostpny...");
                    }
                }
                catch(Exception e)
                {
                    Console.WriteLine(e.ToString());
                    return false;
                }
                return true;
            }

            public static bool IsThreadAvailable(bool checkWorkerThreads)
            {
                int workerThreads = 0;
                int completionPortThreads = 0;
                // odczytanie dostpnych wtkw
                ThreadPool.GetAvailableThreads(out workerThreads,
                    out completionPortThreads);

                // wywietlenie liczby dostpnych wtkw
                Console.WriteLine("{0} wtkw roboczych jest obecnie dostpnych w puli wtkw.",
                    workerThreads);

                if(checkWorkerThreads)
                {
                    if(workerThreads > 0)
                        return true;
                }
                else // sprawdzenie, czy dostpne s wtki portw
                {
                    if(completionPortThreads > 0)
                        return true;
                }
                return false;
            }

            static void ThreadProc(Object stateInfo) 
            {
                // wywietlenie informacji, e wtek do czego si przyda
                Console.WriteLine("Wtek {0} uruchomiony...",stateInfo);
                Thread.Sleep(1000);
            }
        }

        #endregion

        #region 18.8 Waiting for all Threads in the Thread Pool to Finish"
        public static void WaitThreadPoolFinish()
        {
            for(int i=0;i<25;i++)
            {
                // have to wait or threadpool never gives out threads to requests
                Thread.Sleep(50);
                // queue thread request
                ThreadPool.QueueUserWorkItem(ThreadProc1,i);
            }
            // have to wait here or the background threads in the thread
            // pool would not run before the main thread exits.
            Console.WriteLine("Main Thread waiting to complete...");
            bool working = true;
            int workerThreads = 0;
            int completionPortThreads = 0;
            int maxWorkerThreads = 0;
            int maxCompletionPortThreads = 0;
            // get max threads in the pool
            ThreadPool.GetMaxThreads(out maxWorkerThreads,out maxCompletionPortThreads);
            while(working)
            {
                // get available threads
                ThreadPool.GetAvailableThreads(out workerThreads,out completionPortThreads);
                if(workerThreads == maxWorkerThreads)
                {
                    // allow to quit
                    working = false;
                }
                else
                {
                    // sleep before checking again
                    Thread.Sleep(500);
                }
            }
            Console.WriteLine("Main Thread completing...");
        }

        static void ThreadProc1(Object stateInfo) 
        {
            // show we did something with this thread
            Console.WriteLine("Thread {0} running...",stateInfo);
            Thread.Sleep(1000);
        }
        #endregion

        #region 18.8 Konfigurowanie licznika czasu
        public static void ConfigTimer()
        {
            TestTimers.Run();
        }

        public class TestTimers
        {
            public static int count = 0;
            public static Timer timerRef = null;
            private static bool limitHit = false;
            private static object syncRoot = new object();

            public static bool LimitHit
            {
                get
                {
                    lock (syncRoot)
                    {
                        return limitHit;
                    }
                }
                set
                {
                    lock (syncRoot)
                    {
                        limitHit = value;
                    }
                }
            }

            public static void Run()
            {
                TimerCallback callback = TimerMethod;

                // utworzenie licznika czasu, ktry czeka przez p sekundy,
                // a nastpnie jest wywoywany zwrotnie co sekund
                Timer timer = new Timer(callback, null,500, 1000);

                // zapisanie odwoania do licznika, aby wywoanie zwrotne mogo z niego korzysta
                timerRef = timer;

                // gwny wtek niczego nie robi do czasu zakoczenia pracy licznika
                while (true)
                {
                    if (LimitHit == false)
                        Thread.Sleep(0);
                    else
                        break;
                }
                Console.WriteLine("Przykadowy licznik czasu zakoczy prac.");
            }

            static void TimerMethod(Object state)
            {
                count++;
                if(count == 5)
                {
                    LimitHit = true;
                    timerRef.Dispose();
                }
            }
        }

        #endregion

        #region 18.9 Bezpieczne przechowywanie danych wtku
        public static void StoreThreadDataPrivately()
        {
            HandleClass.Run();
        }

        public class Data
        {
            // przechowywanie danych aplikacji
            public int foo;
        }

        public class HandleClass
        {
            public static void Run()
            {
                // utworzenie egzemplarza struktury i zapisanie jej w nazwanym gniedzie danych
                Data appData = new Data();
                Thread.SetData(Thread.GetNamedDataSlot("appDataSlot"), appData);

                // wywoanie kolejnej metody, ktra bdzie uywa struktury
                HandleClass.MethodB();

                // po wykonaniu pracy mona zwolni gniazdo danych
                Thread.FreeNamedDataSlot("appDataSlot");
            }

            public static void MethodB()
            {
                // pobranie egzemplarza struktury z nazwanego gniazda danych
                Data storedAppData = (Data)Thread.GetData(
                    Thread.GetNamedDataSlot("appDataSlot"));

                // modyfikacja struktury StoredAppData

                // po zakoczeniu wprowadzania zmian w strukturze trzeba j z powrotem
                // zapisa w nazwanym gniedzie danych
                Thread.SetData(Thread.GetNamedDataSlot("appDataSlot"), 
                    storedAppData);

                // wywoanie kolejnej metody, ktra bdzie uywa struktury
                HandleClass.MethodC();
            }

            public static void MethodC()
            {
                // pobranie egzemplarza struktury z nazwanego gniazda danych
                Data storedAppData = 
                    (Data)Thread.GetData(Thread.GetNamedDataSlot("appDataSlot"));

                // modyfikacja struktury StoredAppData

                // po zakoczeniu wprowadzania zmian w strukturze trzeba j z powrotem
                // zapisa w nazwanym gniedzie danych
                Thread.SetData(Thread.GetNamedDataSlot("appDataSlot"), storedAppData);
            }
        }

        #endregion

        #region 18.10 Przydzielanie dostpu do zasobu wicej ni jednemu klientowi przy uyciu semafora
        // Xbox posiada 4 porty, do ktrych chce mie dostp grupa programistw.
        // Pocztkowo dostp otrzymuje czterech graczy.
        // Gracze trac yie po upywie losowego czasu, a nowi gracze doczaj do kolejki
        // po zakoczeniu oczekiwania na semaforze.

		public class XboxPlayer
		{
			public class Data
			{
				public ManualResetEvent _GameOver;
				public string _handle;
			}

			// tryby mierci graczy
			private static string[] _deaths = new string[7]{"kupi domek na wsi",
																"udusi si w rakiecie",
																"strzeli sobie w stop",
																"zosta pojmany",
																"ponis mier",
																"zatru si oowiem",
																"nie zdy uskoczy przed granatem",
																};

			/// <summary>
			/// Funkcja Thread
			/// </summary>
			/// <param name="info">Element XboxPlayer.Data z odwoaniem i uchwytem do Xbox</param>
			public static void JoinIn(object info)
			{
				// otwarcie nazwanego semaforu, by mc na nim operowa
                using (Semaphore Xbox = Semaphore.OpenExisting("Xbox"))
                {

                    // pobranie obiektu z danymi
                    Data data = (Data)info;

                    // kady gracz zgasza konsoli ch rozpoczcia gry
                    Console.WriteLine("{0} oczekuje na rozpoczcie gry!", data._handle);

                    // gracze czekaj na konsoli Xbox (na semaforze), a uzyskaj
                    // pozwolenie na zajcie kontrolera                    
                    Xbox.WaitOne();

                    // Xbox wybra gracza! (albo semafor udzieli dostpu do zasobu)
                    Console.WriteLine("{0} zosta wybrany do wzicia udziau w grze. " +
                        "Poznaj swoje przeznaczenie {0}. >:)", data._handle);

                    // wygenerowanie losowego czasu, przez ktry gracz bdzie y
                    System.Random rand = new Random(500);
                    int timeTillDeath = rand.Next(100, 1000);

                    // symulacja gry do czasu utraty ycia
                    Thread.Sleep(timeTillDeath);

                    // ustalenie sposobu utraty ycia
                    rand = new Random();
                    int deathIndex = rand.Next(6);

                    // informacja o wymianie graczy
                    Console.WriteLine("{0} {1} i zrobi miejsce dla kolejnego gracza",
                        data._handle, _deaths[deathIndex]);

                    // jeeli wszystkie porty s otwarte, oznacza to, e wszyscy ju grali i gra si zakoczya
                    int semaphoreCount = Xbox.Release();
                    if (semaphoreCount == 3)
                    {
                        Console.WriteLine("Dzikuj za gr. Gra dobiega koca.");
                        // ustawienie zdarzenia Game Over
                        data._GameOver.Set();
                        // zamknicie semafora
                        Xbox.Close();
                    }
                }
			}
		}		
 
		public class Halo2Session
		{
			// semafor, ktry symuluje ograniczon pul zasobw
			//
			private static Semaphore _Xbox;

			// uchwyty graczy konsoli Xbox
			private static string [] _handles = new string [9]{"Igor",
																"Superman",
																"Zmarzlak",
																"Dr mier",
																"Obozowicz",
																"Egzekutor",
																"abojad",
																"Programista",
																"Niepokonany"
																};



			public static void Play()
			{
				// Xbox posiada 4 porty kontrolera, wic gra moe do czterech graczy naraz.
                // 4 to maksymalna liczba graczy, a zaczynamy od zera, poniewa graczy powinni
                // najpierw utworzy kolejk i poczeka na uruchomienie konsoli i zaadowanie gry
				//
                using (_Xbox = new Semaphore(0, 4, "Xbox"))
                {
                    using (ManualResetEvent GameOver =
                        new ManualResetEvent(false))
                    {

                        //
                        // zalogowanie 9 graczy do gry
                        //
                        for (int i = 0; i < 9; i++)
                        {
                            Thread t = new Thread(XboxPlayer.JoinIn);

                            XboxPlayer.Data playerData = 
                                new XboxPlayer.Data();
                            // ustawienie uchwytu
                            playerData._handle = _handles[i];
                            // ustawienie zdarzenia koca gry
                            playerData._GameOver = GameOver;

                            // zapisanie imienia w wtku
                            t.Name = _handles[i];
                            // uruchomienie gry dla gracza
                            t.Start(playerData);
                        }

                        // oczekiwanie na Xbox na zaadowanie gry Halo2 (3 sekundy)
                        Console.WriteLine("Inicjalizacja konsoli Xbox...");
                        Thread.Sleep(3000);
                        Console.WriteLine(
                            "Halo2 zaadowana i gotowa, udzielenie dostpu czterem graczom...");

                        // Konsola Xbox posiada peen semafor. Wywoywana jest
                        // metoda Release(4), aby otworzy 4 gniazda i umoliwi
                        // maksymalnie czterem oczekujcym graczom na wejcie do konsoli Xbox(semaphore).                        
                        //
                        _Xbox.Release(4);

                        // oczekiwanie na zakoczenie gry...
                        GameOver.WaitOne();
                    }
                }
			}
		}	
		#endregion

        #region 18.11 Synchronizowanie wielu procesw przy uyciu muteksu
        // patrz projekt MutexFun
		#endregion

        #region 18.12 Zapewnienie wsppracy midzy wtkami za pomoc zdarze
        public static void TestResetEvent()
		{
			// posiki przygotowuje kucharz, ktry realizuje tylko jedno zamwienie naraz
			Cook Mel = new Cook();

			// zdefiniowanie piciu kelnerek i nakazanie im, by zbieray zamwienia
			for (int i = 0; i < 5; i++)
			{
				Thread t = new Thread(Waitress.PlaceOrder);
				// kelnerka skada zamwienie i czeka na jego realizacj
				t.Start(Cook.OrderReady);
			}

			// teraz mona zacz wpuszcza klientw
			for (int i = 0; i < 5; i++)
			{
				// kelnerki czekaj...
				Thread.Sleep(2000);
				// ok, kolejna kelnerka!
				Mel.CallWaitress();
			}
		}

		public class Cook
		{
			public static AutoResetEvent OrderReady = new AutoResetEvent(false);

			public void CallWaitress()
			{
				// Wywoujemy Set na AutoResetEvent i nie musimy wywoywa
				// Reset, jak w ManualResetEvent, by ponownie uruchomi zdarzenie.
				// W ten sposb w GetInLine ustawiane jest zdarzenie oznaczajce,
				// e kelnerka oczekuje na zamwienie.
				OrderReady.Set();
			}
		}

		public class Waitress
		{
			public static void PlaceOrder(object signal)
			{
				// zrzutowanie AutoResetEvent, aby kelnerka moga oczekiwa
				// na przygotowanie zamwienia
				AutoResetEvent OrderReady = (AutoResetEvent)signal;
				// oczekiwanie na zamwienie...
				OrderReady.WaitOne();
				// zamwienie zostao zrealizowane....
				Console.WriteLine("Zamwienie kelnerki gotowe!");
			}
		}

		#endregion

        #region 18.13 Uzyskiwanie moliwoci nadawania nazw wasnym zdarzeniom
        public static void TestManualNamedEvent()
		{
            // utworzenie nazwanego ManualResetEvent
            EventWaitHandle ewhSuperBowl = 
                new EventWaitHandle(false, // pocztkowo nie bdzie wysya sygnaw
                                    EventResetMode.ManualReset, 
                                    @"Champs");

            // podpicie trzech wtkw nasuchujcych zdarzenia
            for (int i = 0; i < 3; i++)
            {
	            Thread t = new Thread(ManualFan);
	            // kibice czekaj w napiciu...
                t.Name = "Fan " + i;
	            t.Start();
            }
            // rozgrywamy mecz
            Thread.Sleep(10000);
            // wysanie informacji do kibicw
            Console.WriteLine("Druyna Patriots wygraa SuperBowl!");
            // sygna do wszystkich kibicw
            ewhSuperBowl.Set();
            // zamknicie zdarzenia
            ewhSuperBowl.Close();

		}

        public static void TestAutoNamedEvent()
        {
            // utworzenie nazwanego AutoResetEvent
            EventWaitHandle ewhSuperBowl =
                new EventWaitHandle(false, // pocztkowo nie bdzie wysya sygnaw
                                    EventResetMode.AutoReset,
                                    @"Champs");

            // podpicie trzech wtkw nasuchujcych zdarzenia
            for (int i = 0; i < 3; i++)
            {
                Thread t = new Thread(AutoFan, i);
                // kibice czekaj w napiciu...
                t.Name = "Fan " + i;
                t.Start();
            }
            // rozgrywamy mecz
            Thread.Sleep(10000);
            // wysanie informacji do kibicw
            Console.WriteLine("Druyna Patriots wygraa SuperBowl!");
            // sygna do jednego kibica naraz
            for (int i = 0; i < 3; i++)
            {
                Console.WriteLine("Powiadomienie kibicw");
                ewhSuperBowl.Set();
            }
            // zamknicie zdarzenia
            ewhSuperBowl.Close();
        }


        public static void ManualFan()
        {
	        // otwarcie nazwanego zdarzenia
            EventWaitHandle ewhSuperBowl = 
                new EventWaitHandle(false, 
                                    EventResetMode.ManualReset, 
                                    @"Champs");

	        // oczekiwanie na sygna
            ewhSuperBowl.WaitOne();
	        // okrzyk radoci
	        Console.WriteLine("\"Hurra!\" zawoa {0}",Thread.CurrentThread.Name);
	        // zamknicie zdarzenia
            ewhSuperBowl.Close();
        }

        public static void AutoFan()
        {
            // otwarcie nazwanego zdarzenia
            EventWaitHandle ewhSuperBowl =
                new EventWaitHandle(false,
                                    EventResetMode.ManualReset,
                                    @"Champs");

            // oczekiwanie na sygna
            ewhSuperBowl.WaitOne();
            // okrzyk radoci
            Console.WriteLine("\"Jeeeest!\" zawoa {0}", Thread.CurrentThread.Name);
            // zamknicie zdarzenia
            ewhSuperBowl.Close();
        }
		#endregion

        #region 18.14 Wykonywanie atomowych operacji wrd wtkw
        public static void TestInterlocked()
		{
			int i = 0;
			long l = 0;
			Interlocked.Increment(ref i); // i = 1
			Interlocked.Decrement(ref i); // i = 0
			Interlocked.Increment(ref l); // l = 1
			Interlocked.Decrement(ref i); // l = 0

			Interlocked.Add(ref i, 10); // i = 10;
			Interlocked.Add(ref l, 100); // l = 100;

			string name = "Kajko";
			Interlocked.Exchange(ref name, "Kokosz");

			double runningTotal = 0.0;
			double startingTotal = 0.0;
			double calc = 0.0;
			for (i = 0; i < 10; i++)
			{
				do
				{
					// zapisanie wartoci pocztkowej
					startingTotal = runningTotal;
					// wykonanie zoonych oblicze
					calc = runningTotal + i * Math.PI * 2 / Math.PI;
				}
				// Upewnienie si, e runningTotal nie ulega zmianie
				// i zastpienie jej w takim przypadku wynikiem oblicze. Jeli runningTotal 
				// zostaa zmieniona, odnalezienie w ptli jej wartoci biecej.
				while (startingTotal !=
					Interlocked.CompareExchange(
						ref runningTotal, calc, startingTotal));
			}

			// mniej wiarygodny dom aukcyjny
			Auction Shadys = new Auction("Shady's");
			// wiarygodny dom aukcyjny
			Auction UpperCrust = new Auction("UpperCrust");

			// utworzenie 3 "bezlitosnych" nieznanych licytujcych i 3 nieznanych "bezpiecznych" licytujcych,
			// utworzenie 3 "bezlitosnych " powanych licytujcych i 3 powanych "bezpiecznych" licytujcych
			Thread [] handles = new Thread[12];
			for (i = 0; i < 12; i++)
			{
				Thread t = null;
				switch(i)
				{
					// nieznani bezlitoni licytujcy
					case 0:
					case 1:
					case 2:
					{
						t = new Thread(new ThreadStart(Shadys.Bid));			
					}
					break;
					// nieznani bezpieczni licytujcy
					case 3:
					case 4:
					case 5:
					{
						t = new Thread(new ThreadStart(UpperCrust.BidSafe));			
					}
					break;
					// powani bezlitoni licytujcy
					case 6:
					case 7:
					case 8:
					{
						t = new Thread(new ThreadStart(Shadys.BigBid));			
					}
					break;
					// powani bezpieczni licytujcy
					case 9:
					case 10:
					case 11:
					{
						t = new Thread(new ThreadStart(UpperCrust.BigBidSafe));			
					}
					break;
				}
				// zachowanie odwoania do wtku w celu jego pniejszego uycia
				handles[i] = t;

				// uruchomienie wtku
				t.Start();
			}

			// doczenie do kadego wtku, by poczeka na zakoczenie wszystkich wtkw
			for(i=0;i<12;i++)
				handles[i].Join();

			Console.WriteLine(
				"Dom aukcyjny Shady's zawar {0} transakcji na czn sum ${1} dolarw",
				Shadys._bidCount,Shadys._bidTotal);
			Console.WriteLine(
                "Dom aukcyjny UpperCrust zawar {0} transakcji na czn sum ${1} dolarw",
				UpperCrust._bidCount,UpperCrust._bidTotal);
		}

		/// <summary>
		/// Klasa symulujca dziaanie domu aukcyjnego
		/// </summary>
		public class Auction
		{
			#region Dane publiczne
			// liczba zawartych transakcji
			public int _bidCount = 0;
			// czna liczba transakcji
			public long _bidTotal = 0;
			#endregion

			#region Dane prywatne
			// nazwa domu aukcyjnego
			private string _name;
			#endregion

			#region CTOR
			/// <summary>
			/// Konstruktor, ktry przyjmuje nazw
			/// </summary>
			/// <param name="name">nazwa domu aukcyjnego</param>
			public Auction(string name)
			{
				_name = name;
			}
			#endregion

			#region Metody prywatne
			/// <summary>
			/// Odczekanie losowo wybranego czasu od 0 do 2 sekund
			/// </summary>
			private void Wait()
			{
				// odczekanie losowo wybranego czasu i wykonanie zadania
				Random r = new Random();
				int milli = r.Next(2000);
				Thread.Sleep(milli);
			}
			#endregion

			#region Publiczne metody obsugi wtkw
			public void Bid()
			{
                // odczekanie losowo wybranego czasu przed wykonaniem zadania
				Wait();
				
				// zawarcie transakcji o wartoci 1 bez zwracania uwagi na bezpieczestwo wtkw				
				_bidCount++;
				_bidTotal++;

				// informacja o licytacji
				Console.WriteLine(_name + " zawiera transakcj w wtku " + Thread.CurrentThread.ManagedThreadId);
			}

			public void BidSafe()
			{
                // odczekanie losowo wybranego czasu przed wykonaniem zadania
				Wait();

                // zawarcie transakcji o wartoci 1 z uyciem wtkw
				Interlocked.Increment(ref _bidCount);
				Interlocked.Increment(ref _bidTotal);

                // informacja o licytacji
                Console.WriteLine(_name + " zawiera bezpieczn transakcj w wtku " + Thread.CurrentThread.ManagedThreadId);
			}

			public void BigBid()
			{
                // odczekanie losowo wybranego czasu przed wykonaniem zadania
				Wait();

                // zawarcie transakcji o wartoci 10 bez zwracania uwagi na bezpieczestwo wtkw
				_bidCount += 1;
				_bidTotal += 10;

                // informacja o licytacji
                Console.WriteLine(_name + " zawiera du transakcj w wtku " + Thread.CurrentThread.ManagedThreadId);
			}

			public void BigBidSafe()
			{
                // odczekanie losowo wybranego czasu przed wykonaniem zadania
				Wait();

                // zawarcie transakcji o wartoci 10 z uyciem wtkw
				Interlocked.Add(ref _bidCount, 1);
				Interlocked.Add(ref _bidTotal,10);

                // informacja o licytacji
                Console.WriteLine(_name + " zawiera du bezpieczn transakcj w wtku " + Thread.CurrentThread.ManagedThreadId);
			}
			#endregion
		}
		#endregion

	}
}
